﻿ 
using System;
using System.Linq;
using System.Collections.Generic;

namespace Framework
{
    public static class IEnumerableExtension
    {
        /// <summary>
        /// 非ジェネリックなIEnumerableをジェネリックなIEnumerable<T>に変換します。
        /// </summary>
        public static IEnumerable<T> AsEnumerableT<T>(this System.Collections.IEnumerable target)
        {
            foreach (T t in target)
                yield return t;
        }
    
        /// <summary>
        /// インデックスが取得できるコレクションでラップして返します。
        /// </summary>
        public static DataCollection<T> WithIdx<T>(this IEnumerable<T> target)
        {
            return new DataCollection<T>(target);
        }
        
        /// <summary>
        /// すべての要素にアクションを適用します。
        /// </summary>
        public static void Foreach<T>(this IEnumerable<T> target, Action<T> act)
        {
            foreach (T t in target) act(t);
        }

        /// <summary>
        /// すべての要素にアクションを適用します。
        /// </summary>
        public static void Foreach<T>(this IEnumerable<T> target, Action<T, int> act)
        {
            int i = 0;
            foreach (T t in target)
            {
                act(t, i);
                i++;
            }
        }

        /// <summary>
        /// すべての要素に処理を適用し、最後の処理の結果を返します。
        /// </summary>
        public static R Foreach<T, R>(this IEnumerable<T> target, Func<T, R> func)
        {
            var result = default(R);
            foreach (T t in target) result = func(t);
            return result;
        }

        #region zipメソッド

        /// <summary>
        /// 対象と引数を2要素のタプルにまとめます。
        /// </summary>
        public static IEnumerable<Tuple<T1, T2>> Zip<T1, T2>(this IEnumerable<T1> t1, IEnumerable<T2> t2)
        {
            using (var itr1 = t1.GetEnumerator())
            using (var itr2 = t2.GetEnumerator())
            {
                while (itr1.MoveNext() && itr2.MoveNext())
                    yield return Tuple.Create(itr1.Current, itr2.Current);
            }
        }

        /// <summary>
        /// 対象と引数を3要素のタプルにまとめます。
        /// </summary>
        public static IEnumerable<Tuple<T1, T2, T3>> Zip<T1, T2, T3>(this IEnumerable<T1> t1, IEnumerable<T2> t2, IEnumerable<T3> t3)
        {
            using (var itr1 = t1.GetEnumerator())
            using (var itr2 = t2.GetEnumerator())
            using (var itr3 = t3.GetEnumerator())
            {
                while (itr1.MoveNext() && itr2.MoveNext() && itr3.MoveNext())
                    yield return Tuple.Create(itr1.Current, itr2.Current, itr3.Current);
            }
        }

        /// <summary>
        /// 対象と引数を4要素のタプルにまとめます。
        /// </summary>
        public static IEnumerable<Tuple<T1, T2, T3, T4>> Zip<T1, T2, T3, T4>(this IEnumerable<T1> t1, IEnumerable<T2> t2, IEnumerable<T3> t3, IEnumerable<T4> t4)
        {
            using (var itr1 = t1.GetEnumerator())
            using (var itr2 = t2.GetEnumerator())
            using (var itr3 = t3.GetEnumerator())
            using (var itr4 = t4.GetEnumerator())
            {
                while (itr1.MoveNext() && itr2.MoveNext() && itr3.MoveNext() && itr4.MoveNext())
                    yield return Tuple.Create(itr1.Current, itr2.Current, itr3.Current, itr4.Current);
            }
        }

        /// <summary>
        /// 対象と引数を5要素のタプルにまとめます。
        /// </summary>
        public static IEnumerable<Tuple<T1, T2, T3, T4, T5>> Zip<T1, T2, T3, T4, T5>(this IEnumerable<T1> t1, IEnumerable<T2> t2, IEnumerable<T3> t3, IEnumerable<T4> t4, IEnumerable<T5> t5)
        {
            using (var itr1 = t1.GetEnumerator())
            using (var itr2 = t2.GetEnumerator())
            using (var itr3 = t3.GetEnumerator())
            using (var itr4 = t4.GetEnumerator())
            using (var itr5 = t5.GetEnumerator())
            {
                while (itr1.MoveNext() && itr2.MoveNext() && itr3.MoveNext() && itr4.MoveNext() && itr5.MoveNext())
                    yield return Tuple.Create(itr1.Current, itr2.Current, itr3.Current, itr4.Current, itr5.Current);
            }
        }

        #endregion

    }

    public sealed class Data<T>
    {
        readonly T data;
        readonly int i;
        internal Data(T data, int i)
        {
            this.data = data;
            this.i = i;
        }

        public static implicit operator T(Data<T> d)
        {
            return d.data;
        }

        public int _Index { get { return i; } }
        public T _Data { get { return data; } }
    }

    public sealed class DataCollection<T> : IEnumerable<Data<T>>
    {
        readonly IEnumerable<T> e;

        internal DataCollection(IEnumerable<T> e)
        {
            this.e = e;
        }

        #region IEnumerable<T> メンバ

        public IEnumerator<Data<T>> GetEnumerator()
        {
            int i = 0;
            foreach (T t in e)
            {
                yield return new Data<T>(t, i);
                i++;
            }
        }

        #endregion

        #region IEnumerable メンバ

        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            return ((DataCollection<T>)this).GetEnumerator();
        }

        #endregion
    }
}
